/***************************************************************
* Conduct Kansas Sentencing Regressions, QJPS Version                                        
* Authors: Sanford Gordon, Gregory Huber                   
* Update: March 1, 2007                                   
* Uses:
*    QJPS_activekssentences.dta
*    QJPS_judge_ProxData.dta
*    QJPS_KansasDistrictCharacteristics.dta
*    QJPS_judgeid_sentdate_whetherjudgechallenged.dta
*
* This do file creates:
*  Table 2
*  Table 3, EXCEPT columns (3) and (6)
*  Table 5
*
***************************************************************/

#delimit ;
pause off ;

set mem 400m;
set seed 666;
set matsize 5000;
set more off;

log using "logs\QJPS_BaseReplication.log", replace;

*************
* LOAD DATA *
*************;

*load sentencing data;
use data\QJPS_activekssentences.dta;

************************
* Pre-Process KSC data *
************************;

*Drop murders;
drop if c_dead==1;
drop c_dead;

*Drop offgrid (non-guideline) crimes;
drop if off_non ~= .;

*In the KSC data there is a censoring problem with addcount, such that when addcount=5, counts can be greater than 6;
*We recode addcount to fix this;
replace addcount=(counts-1) if addcount==5 & counts>6;

*Create a binary dependent variable which is 1 if any prison time is assigned;
gen prison=0;
replace prison=1 if totpris>0;

****************************
* End Pre-Process KSC data *
****************************;

*merge in electoral proximity data by judgeid and day of sentencing;
*elecprox is scaled 0 to 1 as described in text;
*alternative measure discussed in footnote 21 is variable proximity_filingdate;
rename sentdate day;
sort judgeid day;
joinby judgeid day using data\QJPS_judge_ProxData.dta;
rename day sentdate;

*merge in district characteristics by district;
sort district;
joinby district using data\QJPS_KansasDistrictCharacteristics.dta;

*merge in data on whether judge was challenged;
*variable challenged=1 if a judge (in a partisan district) is challenged during that term;
sort judgeid sentdate;
joinby judgeid sentdate using data\QJPS_judgeid_sentdate_whetherjudgechallenged.dta;

************************************************
* Create Electoral Proximity Interaction Terms *
************************************************;

*Proximity in competitive and retention districts;
gen comprox = elecprox*(partisan);
gen ncomprox = elecprox*(1-partisan);

*Robustness checks, interactions between certain variables and electoral proximity;
gen crimerate_prox=crimerate*elecprox;
gen sctyesprop_prox=sctyesprop*elecprox;
gen turnout_prox=turnoutpres00*elecprox;

******************************
* Create Judge Fixed Effects *
******************************;

quietly tab judgeid, gen(jd_);

****************
* ANALYZE DATA *
****************;

**************************************
* Create Table 2, summary statistics *
* Note: We first run a silly OLS     *
* regression to set the e(sample) to *
* include only observations with     *
* sentencing data in the summary     *
* stats table.                       *
**************************************;

quietly regress lntotpris ln_gpres0_pris grid1 grid2 addcount pso firearm v_govoff v_child c_assault c_crimthreat c_robbery c_sex c_theft c_burglary appcouns pleabarg male nonwhite hispan age turnoutpres00 presd00prop sctyesprop crimerate nwrate urbanrate partisan elecprox y_*;
summ lntotpris ln_gpres0_pris grid1 grid2 addcount pso firearm v_govoff v_child c_assault c_crimthreat c_robbery c_sex c_theft c_burglary appcouns pleabarg male nonwhite hispan age turnoutpres00 presd00prop sctyesprop crimerate nwrate urbanrate partisan elecprox if e(sample), separator(0);
pause;

*****************************************
* Create Table 3, columns 1,2, 4, and 5 *
* Note: Columns (3) and (6) are created *
* by QJPS_WooldridgeReplication.do      *
*****************************************;

*****************************************************************
*Column (1), basic systemic effect, no district characteristics *
*****************************************************************;

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan, nocons cluster(judgeid) robust;

outreg ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat
c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 partisan using "logs\QJPS_Table3.out", addstat(sigma, e(sigma), log-likelihood, e(ll)) bfmt(fc) bdec(3) 
tdec(3) ctitle("(1)") se nonotes noaster replace;

/***********************************************************
 This simulator estimates quantities of interest 
***********************************************************/

/*Get regression coefficients and variances*/
matrix b = e(b);
matrix v = e(V);

drawnorm b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 b22 b23 b24 b25 b26 b27 b28 b29 sigma, means(b) cov(v);
replace sigma = exp(sigma);

mkmat b1-b29 in 1/2000, matrix(beta);
mkmat sigma in 1/2000, matrix(sigma);

matrix prob_ret = J(2000,1,0);
matrix prob_com = J(2000,1,0);
matrix es_ret = J(2000,1,0);
matrix es_com = J(2000,1,0);
	
matrix prob_ret_fd = J(2000,1,0);
matrix prob_com_fd = J(2000,1,0);
matrix es_ret_fd = J(2000,1,0);
matrix es_com_fd = J(2000,1,0);

/*
 create variable value vectors.
*/

/* 
For the probability that someone goes to prison, we set all variables at their sample medians, 
 except dummy variables, which are binary or dominant category in sample.
For district characteristics, we used median in sample.
This means we are looking at burglaries in 2001.
*/

matrix xr_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0;
matrix xr_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\1;

/* 
For sentence length, we keep all other variables as above but change the baseline sentence
 to the median among felons who get any	prison time
*/

matrix xr_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0;
matrix xr_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\1;

/*
z's are beta's times values
*/
	
matrix z_ret_pr = beta*xr_ret_pr;
matrix z_com_pr = beta*xr_com_pr;
matrix z_ret_es = beta*xr_ret_es;
matrix z_com_es = beta*xr_com_es;

/*
loop over simulated betas, store results in results matrices
*/

local i = 1;
while `i' <= 2000 {;
	scalar p_r = normprob(z_ret_pr[`i',1]/sigma[`i',1]);
	matrix prob_ret[`i',1] = p_r;
	scalar p_c = normprob(z_com_pr[`i',1]/sigma[`i',1]);
	matrix prob_com[`i',1] = p_c;
	scalar e_r = z_ret_es[`i',1]+sigma[`i',1]*normden(-z_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_ret_es[`i',1]/sigma[`i',1]));
	matrix es_ret[`i',1] = e_r;
	scalar e_c = z_com_es[`i',1]+sigma[`i',1]*normden(-z_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_com_es[`i',1]/sigma[`i',1]));
	matrix es_com[`i',1] = e_c;
	local i = `i'+1;
	};

/*
convert simulation results matrices into variables
*/

svmat prob_ret, names(prob_ret_v);
svmat prob_com, names(prob_com_v);
svmat es_ret, names(es_ret_v);
svmat es_com, names(es_com_v);

/*
calculate observed difference in probabilities of prison
*/

gen probdif = prob_com_v1-prob_ret_v1;

/*
Calculate the change in the odds ratio
*/

gen odds_com = prob_com_v1/(1-prob_com_v1);
gen odds_ret = prob_ret_v1/(1-prob_ret_v1);
gen oddsdif = (odds_com-odds_ret)/odds_ret;

/*
Calculate the change in the sentence
*/

replace es_com_v1=exp(es_com_v1);
replace es_ret_v1=exp(es_ret_v1);

gen sentdif = es_com_v1-es_ret_v1;

/*
Take 2.5, 50, and 97.5th percentile of quantities of interest
*/

centile probdif, centile(2.5 50 97.5);

centile prob_ret_v1;

centile oddsdif, centile(2.5 50 97.5);

centile odds_ret;

centile sentdif, centile(2.5 50 97.5);

centile es_ret;

/*
drop simulator variables for next loop
*/

drop prob_ret_v1-sentdif;

drop b1-sigma;

*******************************************************************
*Column (2), basic systemic effect, with district characteristics *
*******************************************************************;

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan, nocons cluster(judgeid) robust;

/*save regression results*/

outreg ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat
c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 partisan
turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop using "logs\QJPS_Table3.out", addstat(sigma, e(sigma), log-likelihood, e(ll)) bfmt(fc) bdec(3) 
tdec(3) ctitle("(2)") se nonotes noaster append;

/***********************************************************
 This simulator estimates quantities of interest 
***********************************************************/

/*Get regression coefficients and variances*/
matrix b = e(b);
matrix v = e(V);

drawnorm b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 b22 b23 b24 b25 b26 b27 b28 b29 b30 b31 b32 b33 b34 b35 sigma, means(b) cov(v);
replace sigma = exp(sigma);

mkmat b1-b35 in 1/2000, matrix(beta);
mkmat sigma in 1/2000, matrix(sigma);

matrix prob_ret = J(2000,1,0);
matrix prob_com = J(2000,1,0);
matrix es_ret = J(2000,1,0);
matrix es_com = J(2000,1,0);
	
matrix prob_ret_fd = J(2000,1,0);
matrix prob_com_fd = J(2000,1,0);
matrix es_ret_fd = J(2000,1,0);
matrix es_com_fd = J(2000,1,0);

/*
 create variable value vectors.
*/

/* 
For the probability that someone goes to prison, we set all variables at their sample medians, 
 except dummy variables, which are binary or dominant category in sample.
For district characteristics, we used median in sample.
This means we are looking at burglaries in 2001.
*/

matrix xr_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0;
matrix xr_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1;

/* 
For sentence length, we keep all other variables as above but change the baseline sentence
 to the median among felons who get any	prison time
*/

matrix xr_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0;
matrix xr_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1;

/*
z's are beta's times values
*/
	
matrix z_ret_pr = beta*xr_ret_pr;
matrix z_com_pr = beta*xr_com_pr;
matrix z_ret_es = beta*xr_ret_es;
matrix z_com_es = beta*xr_com_es;

/*
loop over simulated betas, store results in results matrices
*/

local i = 1;
while `i' <= 2000 {;
	scalar p_r = normprob(z_ret_pr[`i',1]/sigma[`i',1]);
	matrix prob_ret[`i',1] = p_r;
	scalar p_c = normprob(z_com_pr[`i',1]/sigma[`i',1]);
	matrix prob_com[`i',1] = p_c;
	scalar e_r = z_ret_es[`i',1]+sigma[`i',1]*normden(-z_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_ret_es[`i',1]/sigma[`i',1]));
	matrix es_ret[`i',1] = e_r;
	scalar e_c = z_com_es[`i',1]+sigma[`i',1]*normden(-z_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_com_es[`i',1]/sigma[`i',1]));
	matrix es_com[`i',1] = e_c;
	local i = `i'+1;
	};

/*
convert simulation results matrices into variables
*/

svmat prob_ret, names(prob_ret_v);
svmat prob_com, names(prob_com_v);
svmat es_ret, names(es_ret_v);
svmat es_com, names(es_com_v);

/*
calculate observed difference in probabilities of prison
*/

gen probdif = prob_com_v1-prob_ret_v1;

/*
Calculate the change in the odds ratio
*/

gen odds_com = prob_com_v1/(1-prob_com_v1);
gen odds_ret = prob_ret_v1/(1-prob_ret_v1);
gen oddsdif = (odds_com-odds_ret)/odds_ret;

/*
Calculate the change in the sentence
*/

replace es_com_v1=exp(es_com_v1);
replace es_ret_v1=exp(es_ret_v1);

gen sentdif = es_com_v1-es_ret_v1;

/*
Take 2.5, 50, and 97.5th percentile of quantities of interest
*/

centile probdif, centile(2.5 50 97.5);

centile prob_ret_v1;

centile oddsdif, centile(2.5 50 97.5);

centile odds_ret;

centile sentdif, centile(2.5 50 97.5);

centile es_ret;

/*
drop simulator variables for next loop
*/

drop prob_ret_v1-sentdif;

drop b1-sigma;

*******************************************************************************************
*Column (4), basic systemic effect, no district characteristics, UNCHALLENGED judges only *
*******************************************************************************************;

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan if challenged==0, nocons cluster(judgeid) robust;

/*save regression results*/

outreg ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat
c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 partisan using "logs\QJPS_Table3.out", addstat(sigma, e(sigma), log-likelihood, e(ll)) bfmt(fc) bdec(3) 
tdec(3) ctitle("(4)") se nonotes noaster append;

/***********************************************************
 This simulator estimates quantities of interest 
***********************************************************/

/*Get regression coefficients and variances*/
matrix b = e(b);
matrix v = e(V);

drawnorm b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 b22 b23 b24 b25 b26 b27 b28 b29 sigma, means(b) cov(v);
replace sigma = exp(sigma);

mkmat b1-b29 in 1/2000, matrix(beta);
mkmat sigma in 1/2000, matrix(sigma);

matrix prob_ret = J(2000,1,0);
matrix prob_com = J(2000,1,0);
matrix es_ret = J(2000,1,0);
matrix es_com = J(2000,1,0);
	
matrix prob_ret_fd = J(2000,1,0);
matrix prob_com_fd = J(2000,1,0);
matrix es_ret_fd = J(2000,1,0);
matrix es_com_fd = J(2000,1,0);

/*
 create variable value vectors.
*/

/* 
For the probability that someone goes to prison, we set all variables at their sample medians, 
 except dummy variables, which are binary or dominant category in sample.
For district characteristics, we used median in sample.
This means we are looking at burglaries in 2001.
*/

matrix xr_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0;
matrix xr_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\1;

/* 
For sentence length, we keep all other variables as above but change the baseline sentence
 to the median among felons who get any	prison time
*/

matrix xr_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0;
matrix xr_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\1;

/*
z's are beta's times values
*/
	
matrix z_ret_pr = beta*xr_ret_pr;
matrix z_com_pr = beta*xr_com_pr;
matrix z_ret_es = beta*xr_ret_es;
matrix z_com_es = beta*xr_com_es;

/*
loop over simulated betas, store results in results matrices
*/

local i = 1;
while `i' <= 2000 {;
	scalar p_r = normprob(z_ret_pr[`i',1]/sigma[`i',1]);
	matrix prob_ret[`i',1] = p_r;
	scalar p_c = normprob(z_com_pr[`i',1]/sigma[`i',1]);
	matrix prob_com[`i',1] = p_c;
	scalar e_r = z_ret_es[`i',1]+sigma[`i',1]*normden(-z_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_ret_es[`i',1]/sigma[`i',1]));
	matrix es_ret[`i',1] = e_r;
	scalar e_c = z_com_es[`i',1]+sigma[`i',1]*normden(-z_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_com_es[`i',1]/sigma[`i',1]));
	matrix es_com[`i',1] = e_c;
	local i = `i'+1;
	};

/*
convert simulation results matrices into variables
*/

svmat prob_ret, names(prob_ret_v);
svmat prob_com, names(prob_com_v);
svmat es_ret, names(es_ret_v);
svmat es_com, names(es_com_v);

/*
calculate observed difference in probabilities of prison
*/

gen probdif = prob_com_v1-prob_ret_v1;

/*
Calculate the change in the odds ratio
*/

gen odds_com = prob_com_v1/(1-prob_com_v1);
gen odds_ret = prob_ret_v1/(1-prob_ret_v1);
gen oddsdif = (odds_com-odds_ret)/odds_ret;

/*
Calculate the change in the sentence
*/

replace es_com_v1=exp(es_com_v1);
replace es_ret_v1=exp(es_ret_v1);

gen sentdif = es_com_v1-es_ret_v1;

/*
Take 2.5, 50, and 97.5th percentile of quantities of interest
*/

centile probdif, centile(2.5 50 97.5);

centile prob_ret_v1;

centile oddsdif, centile(2.5 50 97.5);

centile odds_ret;

centile sentdif, centile(2.5 50 97.5);

centile es_ret;

/*
drop simulator variables for next loop
*/

drop prob_ret_v1-sentdif;

drop b1-sigma;

*********************************************************************************************
*Column (5), basic systemic effect, with district characteristics, UNCHALLENGED judges only *
*********************************************************************************************;

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan if challenged==0, nocons cluster(judgeid) robust;

/*save regression results*/

outreg ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat
c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 partisan
turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop using "logs\QJPS_Table3.out", addstat(sigma, e(sigma), log-likelihood, e(ll)) bfmt(fc) bdec(3) 
tdec(3) ctitle("(5)") se nonotes noaster append;

/***********************************************************
 This simulator estimates quantities of interest 
***********************************************************/

/*Get regression coefficients and variances*/
matrix b = e(b);
matrix v = e(V);

drawnorm b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 b22 b23 b24 b25 b26 b27 b28 b29 b30 b31 b32 b33 b34 b35 sigma, means(b) cov(v);
replace sigma = exp(sigma);

mkmat b1-b35 in 1/2000, matrix(beta);
mkmat sigma in 1/2000, matrix(sigma);

matrix prob_ret = J(2000,1,0);
matrix prob_com = J(2000,1,0);
matrix es_ret = J(2000,1,0);
matrix es_com = J(2000,1,0);
	
matrix prob_ret_fd = J(2000,1,0);
matrix prob_com_fd = J(2000,1,0);
matrix es_ret_fd = J(2000,1,0);
matrix es_com_fd = J(2000,1,0);

/*
 create variable value vectors.
*/

/* 
For the probability that someone goes to prison, we set all variables at their sample medians, 
 except dummy variables, which are binary or dominant category in sample.
For district characteristics, we used median in sample.
This means we are looking at burglaries in 2001.
*/

matrix xr_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0;
matrix xr_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1;

/* 
For sentence length, we keep all other variables as above but change the baseline sentence
 to the median among felons who get any	prison time
*/

matrix xr_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0;
matrix xr_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1;

/*
z's are beta's times values
*/
	
matrix z_ret_pr = beta*xr_ret_pr;
matrix z_com_pr = beta*xr_com_pr;
matrix z_ret_es = beta*xr_ret_es;
matrix z_com_es = beta*xr_com_es;

/*
loop over simulated betas, store results in results matrices
*/

local i = 1;
while `i' <= 2000 {;
	scalar p_r = normprob(z_ret_pr[`i',1]/sigma[`i',1]);
	matrix prob_ret[`i',1] = p_r;
	scalar p_c = normprob(z_com_pr[`i',1]/sigma[`i',1]);
	matrix prob_com[`i',1] = p_c;
	scalar e_r = z_ret_es[`i',1]+sigma[`i',1]*normden(-z_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_ret_es[`i',1]/sigma[`i',1]));
	matrix es_ret[`i',1] = e_r;
	scalar e_c = z_com_es[`i',1]+sigma[`i',1]*normden(-z_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_com_es[`i',1]/sigma[`i',1]));
	matrix es_com[`i',1] = e_c;
	local i = `i'+1;
	};

/*
convert simulation results matrices into variables
*/

svmat prob_ret, names(prob_ret_v);
svmat prob_com, names(prob_com_v);
svmat es_ret, names(es_ret_v);
svmat es_com, names(es_com_v);

/*
calculate observed difference in probabilities of prison
*/

gen probdif = prob_com_v1-prob_ret_v1;

/*
Calculate the change in the odds ratio
*/

gen odds_com = prob_com_v1/(1-prob_com_v1);
gen odds_ret = prob_ret_v1/(1-prob_ret_v1);
gen oddsdif = (odds_com-odds_ret)/odds_ret;

/*
Calculate the change in the sentence
*/

replace es_com_v1=exp(es_com_v1);
replace es_ret_v1=exp(es_ret_v1);

gen sentdif = es_com_v1-es_ret_v1;

/*
Take 2.5, 50, and 97.5th percentile of quantities of interest
*/

centile probdif, centile(2.5 50 97.5);

centile prob_ret_v1;

centile oddsdif, centile(2.5 50 97.5);

centile odds_ret;

centile sentdif, centile(2.5 50 97.5);

centile es_ret;

/*
drop simulator variables for next loop
*/

drop prob_ret_v1-sentdif;

drop b1-sigma;

*****************************************
* Create Table 5, columns 1 and 2       *
*****************************************;

/* ***************************************************** */
/* Interaction between electoral proximity and partisan */
/* ***************************************************** */

/* ***************************************************************** */
/* Column (1), Model with district characteristics, no judge dummies */
/* ***************************************************************** */

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan comprox ncomprox, nocons cluster(judgeid) robust;

lincom partisan * 1 + comprox*.30 - ncomprox*.30;
lincom partisan * 1 + comprox*.33 - ncomprox*.33;
lincom partisan * 1 + comprox*.35 - ncomprox*.35;
test comprox==ncomprox;

/*save regression results*/

outreg ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat
c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 
turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop partisan comprox ncomprox using "logs\QJPS_Table5.out", addstat(sigma, e(sigma), log-likelihood, e(ll)) bfmt(fc) bdec(3) 
tdec(3) ctitle("(1)") se nonotes noaster replace;

/***********************************************************
 This simulator estimates quantities of interest 
***********************************************************/

/*First, generate variables to store simulator output */

gen proxhat=.;
gen prob_ret_l = .;
gen prob_ret_m = .;
gen prob_ret_h = .;
gen prob_com_l = .;
gen prob_com_m = .;
gen prob_com_h = .;
gen probdif_l = .;
gen probdif_m = .;
gen probdif_h = .;
gen odds_ret_l = .;
gen odds_ret_m = .;
gen odds_ret_h = .;
gen odds_com_l = .;
gen odds_com_m = .;
gen odds_com_h = .;
gen oddsdif_l = .;
gen oddsdif_m = .;
gen oddsdif_h = .;
gen es_ret_l = .;
gen es_ret_m = .;
gen es_ret_h = .;
gen es_com_l = .;
gen es_com_m = .;
gen es_com_h = .;
gen sentdif_l = .;
gen sentdif_m = .;
gen sentdif_h = .;

gen prob_ret_fd_l = .;
gen prob_ret_fd_m = .;
gen prob_ret_fd_h = .;
gen prob_com_fd_l = .;
gen prob_com_fd_m = .;
gen prob_com_fd_h = .;
gen es_ret_fd_l = .;
gen es_ret_fd_m = .;
gen es_ret_fd_h = .;
gen es_com_fd_l = .;
gen es_com_fd_m = .;
gen es_com_fd_h = .;

/*I call this model, without judge-fixed effects, model1*/
gen model=.;
replace model=1 in 1/21;

/*Get regression coefficients and variances*/

matrix b = e(b);
matrix v = e(V);

drawnorm b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 b22 b23 b24 b25 b26 b27 b28 b29 b30 b31 b32 b33 b34 b35 b36 b37 sigma, means(b) cov(v);
replace sigma = exp(sigma);

mkmat b1-b37 in 1/2000, matrix(beta);
mkmat sigma in 1/2000, matrix(sigma);

/********************************************************************
 The simulator needs to loop over all possible values of proximity
 We'll do it in steps of .05 from 0 to 1 by taking our loop variable
 and subtracting 1, and then dividing by 20.
********************************************************************/

local proxloop = 1;
while `proxloop' <= 21 {;

	local proxtemp=(`proxloop'-1)/20;
	local obstemp=(`proxloop');

	disp "Calculating simulations for proximity= " + `proxtemp';

	/*
	zero out results matrices
	*/
	
	matrix prob_ret = J(2000,1,0);
	matrix prob_com = J(2000,1,0);
	matrix es_ret = J(2000,1,0);
	matrix es_com = J(2000,1,0);
	
	matrix prob_ret_fd = J(2000,1,0);
	matrix prob_com_fd = J(2000,1,0);
	matrix es_ret_fd = J(2000,1,0);
	matrix es_com_fd = J(2000,1,0);

	/*
	 create variable value vectors.
	 The proximity variable at the end assigns the appropriate proximity variable
	 (comprox or ncomprox) the appropriate value of proximity
	*/

	/* 
	For the probability that someone goes to prison, we set all variables at their sample means, 
	 except dummy variables, which are binary or dominant category in sample.
	For district characteristics, we used median in sample.
	This means we are looking at burglaries in 2001.
	*/

	matrix xr_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0\0\(`proxtemp');
	matrix xr_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1\(`proxtemp')\0;

	/* 
	For sentence length, we keep all other variables as above but change the baseline sentence
	 to the median among felons who get any	prison time
	*/

	matrix xr_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0\0\(`proxtemp');
	matrix xr_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1\(`proxtemp')\0;

        /*
        These matrices are just used when proxtemp==0, which is when we calculate the overall first difference for shifting prox 0 to 1
        */

        if `proxtemp'==0 {;
		matrix fd_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0\0\1;
		matrix fd_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1\1\0;
		matrix fd_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\0\0\1;
		matrix fd_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\.5384412\.3316583\.0844266\.5555941\25.4378\.7495779\0\0\0\0\1\0\0\1\1\0;
	};
	
	/*
	z's are beta's times values
	*/
	
	matrix z_ret_pr = beta*xr_ret_pr;
	matrix z_com_pr = beta*xr_com_pr;
	matrix z_ret_es = beta*xr_ret_es;
	matrix z_com_es = beta*xr_com_es;

        if `proxtemp'==0 {;
		matrix z_fd_ret_pr = beta*fd_ret_pr;
		matrix z_fd_com_pr = beta*fd_com_pr;
		matrix z_fd_ret_es = beta*fd_ret_es;
		matrix z_fd_com_es = beta*fd_com_es;
	};

	/*
	loop over simulated betas, store results in results matrices
	*/
	
	local i = 1;
	while `i' <= 2000 {;
		scalar p_r = normprob(z_ret_pr[`i',1]/sigma[`i',1]);
		matrix prob_ret[`i',1] = p_r;
		scalar p_c = normprob(z_com_pr[`i',1]/sigma[`i',1]);
		matrix prob_com[`i',1] = p_c;
		scalar e_r = z_ret_es[`i',1]+sigma[`i',1]*normden(-z_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_ret_es[`i',1]/sigma[`i',1]));
		matrix es_ret[`i',1] = e_r;
		scalar e_c = z_com_es[`i',1]+sigma[`i',1]*normden(-z_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_com_es[`i',1]/sigma[`i',1]));
		matrix es_com[`i',1] = e_c;
		if `proxtemp'==0 {;
			scalar p_r_fd = normprob(z_fd_ret_pr[`i',1]/sigma[`i',1]);
			matrix prob_ret_fd[`i',1] = p_r_fd;
			scalar p_c_fd = normprob(z_fd_com_pr[`i',1]/sigma[`i',1]);
			matrix prob_com_fd[`i',1] = p_c_fd;
			scalar e_r_fd = z_fd_ret_es[`i',1]+sigma[`i',1]*normden(-z_fd_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_fd_ret_es[`i',1]/sigma[`i',1]));
			matrix es_ret_fd[`i',1] = e_r_fd;
			scalar e_c_fd = z_fd_com_es[`i',1]+sigma[`i',1]*normden(-z_fd_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_fd_com_es[`i',1]/sigma[`i',1]));
			matrix es_com_fd[`i',1] = e_c_fd;
		};	
		
		local i = `i'+1;
	};

	/*
	convert simulation results matrices into variables
	*/
	
	svmat prob_ret, names(prob_ret_v);
	svmat prob_com, names(prob_com_v);
	svmat es_ret, names(es_ret_v);
	svmat es_com, names(es_com_v);
	
	svmat prob_ret_fd, names(prob_ret_fd_v);
	svmat prob_com_fd, names(prob_com_fd_v);
	svmat es_ret_fd, names(es_ret_fd_v);
	svmat es_com_fd, names(es_com_fd_v);

	/*
	calculate observed difference in probabilities of prison
	*/
	
	gen probdif = prob_com_v1-prob_ret_v1;

	if `proxtemp'==0 {;
		replace prob_ret_fd_v1=prob_ret_fd_v1-prob_ret_v1;
		replace prob_com_fd_v1=prob_com_fd_v1-prob_com_v1;
		};

	/*
	Calculate the change in the odds ratio
	*/
	
	gen odds_com = prob_com_v1/(1-prob_com_v1);
	gen odds_ret = prob_ret_v1/(1-prob_ret_v1);
	gen oddsdif = (odds_com-odds_ret)/odds_ret;

	/*
	Calculate the change in the sentence
	*/

	replace es_com_v1=exp(es_com_v1);
	replace es_ret_v1=exp(es_ret_v1);

	gen sentdif = es_com_v1-es_ret_v1;

	if `proxtemp'==0 {;
		replace es_ret_fd_v1=exp(es_ret_fd_v1);
		replace es_com_fd_v1=exp(es_com_fd_v1);
		replace es_ret_fd_v1=es_ret_fd_v1-es_ret_v1;
		replace es_com_fd_v1=es_com_fd_v1-es_com_v1;
	};
	
	/*
	Take 2.5, 50, and 97.5th percentile of quantities of interest
	*/
	
	qui centile prob_ret_v1, centile(2.5 50 97.5);
	qui replace prob_ret_l = r(c_1) in `obstemp';
	qui replace prob_ret_m = r(c_2) in `obstemp';
	qui replace prob_ret_h = r(c_3) in `obstemp';
	
	qui centile prob_com_v1, centile(2.5 50 97.5);
	qui replace prob_com_l = r(c_1) in `obstemp';
	qui replace prob_com_m = r(c_2) in `obstemp';
	qui replace prob_com_h = r(c_3) in `obstemp';
	
	qui centile probdif, centile(2.5 50 97.5);
	qui replace probdif_l = r(c_1) in `obstemp';
	qui replace probdif_m = r(c_2) in `obstemp';
	qui replace probdif_h = r(c_3) in `obstemp';

	qui centile odds_ret, centile(2.5 50 97.5);
	qui replace odds_ret_l = r(c_1) in `obstemp';
	qui replace odds_ret_m = r(c_2) in `obstemp';
	qui replace odds_ret_h = r(c_3) in `obstemp';
	
	qui centile odds_com, centile(2.5 50 97.5);
	qui replace odds_com_l = r(c_1) in `obstemp';
	qui replace odds_com_m = r(c_2) in `obstemp';
	qui replace odds_com_h = r(c_3) in `obstemp';

	qui centile oddsdif, centile(2.5 50 97.5);
	qui replace oddsdif_l = r(c_1) in `obstemp';
	qui replace oddsdif_m = r(c_2) in `obstemp';
	qui replace oddsdif_h = r(c_3) in `obstemp';

	qui centile es_ret_v1, centile(2.5 50 97.5);
	qui replace es_ret_l = r(c_1) in `obstemp';
	qui replace es_ret_m = r(c_2) in `obstemp';
	qui replace es_ret_h = r(c_3) in `obstemp';
	
	qui centile es_com_v1, centile(2.5 50 97.5);
	qui replace es_com_l = r(c_1) in `obstemp';
	qui replace es_com_m = r(c_2) in `obstemp';
	qui replace es_com_h = r(c_3) in `obstemp';

	qui centile sentdif, centile(2.5 50 97.5);
	qui replace sentdif_l = r(c_1) in `obstemp';
	qui replace sentdif_m = r(c_2) in `obstemp';
	qui replace sentdif_h = r(c_3) in `obstemp';
	
	if `proxtemp'==0 {;
		qui centile prob_ret_fd_v1, centile(2.5 50 97.5);
		qui replace prob_ret_fd_l = r(c_1) in `obstemp';
		qui replace prob_ret_fd_m = r(c_2) in `obstemp';
		qui replace prob_ret_fd_h = r(c_3) in `obstemp';

		qui centile prob_com_fd_v1, centile(2.5 50 97.5);
		qui replace prob_com_fd_l = r(c_1) in `obstemp';
		qui replace prob_com_fd_m = r(c_2) in `obstemp';
		qui replace prob_com_fd_h = r(c_3) in `obstemp';

		qui centile es_ret_fd_v1, centile(2.5 50 97.5);
		qui replace es_ret_fd_l = r(c_1) in `obstemp';
		qui replace es_ret_fd_m = r(c_2) in `obstemp';
		qui replace es_ret_fd_h = r(c_3) in `obstemp';

		qui centile es_com_fd_v1, centile(2.5 50 97.5);
		qui replace es_com_fd_l = r(c_1) in `obstemp';
		qui replace es_com_fd_m = r(c_2) in `obstemp';
		qui replace es_com_fd_h = r(c_3) in `obstemp';
	};	
	
	/*
	drop simulator variables for next loop
	*/

	drop prob_ret_v1-sentdif;
	replace proxhat=`proxtemp' in `obstemp';
	
	local proxloop = `proxloop' + 1;
	
};

drop b1-sigma;

/* summarize first differences shifts in electoral proximity */

/* Probability of incerceration */
l prob_ret_fd* prob_com_fd* in 1;

/* baselines */
l prob_ret_m prob_com_m in 1;

/* Sentence, given any sentence assigned */
l es_ret_fd_* es_com_fd_* in 1;

/* baselines */
l es_ret_m es_com_m in 1;

l proxhat prob_ret_l-prob_com_h if model==1;

l proxhat probdif* if model==1;

l proxhat sentdif* if model==1;

drop proxhat-model;

/* ***************************************************************** */
/* Column (2), Model with judge dummies                              */
/* ***************************************************************** */

/*we estimate the model with jd_70 first before the other judge dummies because that judge is the modal judge*/
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 comprox ncomprox jd_70 jd_1-jd_69 jd_71-jd_135, nocons robust;

test comprox==ncomprox;

/*save regression results*/

outreg ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat
c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 
comprox ncomprox using "logs\QJPS_Table5.out", addstat(sigma, e(sigma), log-likelihood, e(ll)) bfmt(fc) bdec(3) 
tdec(3) ctitle("(2)") se nonotes noaster append;

/***********************************************************
 This simulator estimates quantities of interest 
***********************************************************/

/*First, generate variables to store simulator output */

gen proxhat=.;
gen prob_ret_l = .;
gen prob_ret_m = .;
gen prob_ret_h = .;
gen prob_com_l = .;
gen prob_com_m = .;
gen prob_com_h = .;
gen probdif_l = .;
gen probdif_m = .;
gen probdif_h = .;
gen odds_ret_l = .;
gen odds_ret_m = .;
gen odds_ret_h = .;
gen odds_com_l = .;
gen odds_com_m = .;
gen odds_com_h = .;
gen oddsdif_l = .;
gen oddsdif_m = .;
gen oddsdif_h = .;
gen es_ret_l = .;
gen es_ret_m = .;
gen es_ret_h = .;
gen es_com_l = .;
gen es_com_m = .;
gen es_com_h = .;
gen sentdif_l = .;
gen sentdif_m = .;
gen sentdif_h = .;

gen prob_ret_fd_l = .;
gen prob_ret_fd_m = .;
gen prob_ret_fd_h = .;
gen prob_com_fd_l = .;
gen prob_com_fd_m = .;
gen prob_com_fd_h = .;
gen es_ret_fd_l = .;
gen es_ret_fd_m = .;
gen es_ret_fd_h = .;
gen es_com_fd_l = .;
gen es_com_fd_m = .;
gen es_com_fd_h = .;

/*I call this model, with judge-fixed effects, model2*/
gen model=.;
replace model=2 in 1/21;

/*Get regression coefficients and variances*/

matrix b = e(b);
matrix v = e(V);

drawnorm b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15 b16 b17 b18 b19 b20 b21 b22 b23 b24 b25 b26 b27 b28 b29 b30 b31 b32 b33 b34 b35 b36 b37 b38 b39 b40 b41 b42 b43 b44 b45 b46 b47 b48 b49 b50 b51 b52 b53 b54 b55 b56 b57 b58 b59 b60 b61 b62 b63 b64 b65 b66 b67 b68 b69 b70 b71 b72 b73 b74 b75 b76 b77 b78 b79 b80 b81 b82 b83 b84 b85 b86 b87 b88 b89 b90 b91 b92 b93 b94 b95 b96 b97 b98 b99 b100 b101 b102 b103 b104 b105 b106 b107 b108 b109 b110 b111 b112 b113 b114 b115 b116 b117 b118 b119 b120 b121 b122 b123 b124 b125 b126 b127 b128 b129 b130 b131 b132 b133 b134 b135 b136 b137 b138 b139 b140 b141 b142 b143 b144 b145 b146 b147 b148 b149 b150 b151 b152 b153 b154 b155 b156 b157 b158 b159 b160 b161 b162 b163 b164 sigma, means(b) cov(v);
replace sigma = exp(sigma);

mkmat b1-b31 in 1/2000, matrix(beta);
mkmat sigma in 1/2000, matrix(sigma);

/********************************************************************
 The simulator needs to loop over all possible values of proximity
 We'll do it in steps of .05 from 0 to 1 by taking our loop variable
 and then dividing by 20.
********************************************************************/

local proxloop = 1;
while `proxloop' <= 21 {;

	local proxtemp=(`proxloop'-1)/20;
	local obstemp=(`proxloop');
	disp "Calculating simulations for proximity= " + `proxtemp';

	/*
	zero out results matrices
	*/
	
	matrix prob_ret = J(2000,1,0);
	matrix prob_com = J(2000,1,0);
	matrix es_ret = J(2000,1,0);
	matrix es_com = J(2000,1,0);

	/*
	 create variable value vectors.
	 The proximity variable at the end assigns the appropriate proximity variable
	 (comprox or ncomprox) the appropriate value of proximity
	*/

	/* 
	For the probability that someone goes to prison, we set all variables at their sample means, 
	 except dummy variables, which are binary or dominant category in sample.
	For district characteristics, we used median in sample.
	This means we are looking at burglaries in 2001.
	*/

	matrix xr_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0\(`proxtemp')\1;
	matrix xr_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\(`proxtemp')\0\1;

	/* 
	For sentence length, we keep all other variables as above but change the baseline sentence
	 to the median among felons who get any	prison time
	*/

	matrix xr_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0\(`proxtemp')\1;
	matrix xr_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\(`proxtemp')\0\1;

        /*
        These matrices are just used when proxtemp==0, which is when we calculate the overall first difference for shifting prox 0 to 1
        */

        if `proxtemp'==0 {;
		matrix fd_ret_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0\1\1;
		matrix fd_com_pr = 0\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\1\0\1;
		matrix fd_ret_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\0\1\1;
		matrix fd_com_es = 3.496508\0\0\0\0\0\0\0\0\0\0\1\0\0\1\1\0\1\0\26.28063\690.6715132\0\0\0\0\1\0\0\1\0\1;
	};
	
	/*
	z's are beta's times values
	*/
	
	matrix z_ret_pr = beta*xr_ret_pr;
	matrix z_com_pr = beta*xr_com_pr;
	matrix z_ret_es = beta*xr_ret_es;
	matrix z_com_es = beta*xr_com_es;

        if `proxtemp'==0 {;
		matrix z_fd_ret_pr = beta*fd_ret_pr;
		matrix z_fd_com_pr = beta*fd_com_pr;
		matrix z_fd_ret_es = beta*fd_ret_es;
		matrix z_fd_com_es = beta*fd_com_es;
	};

	/*
	loop over simulated betas, store results in results matrices
	*/
	
	local i = 1;
	while `i' <= 2000 {;
		scalar p_r = normprob(z_ret_pr[`i',1]/sigma[`i',1]);
		matrix prob_ret[`i',1] = p_r;
		scalar p_c = normprob(z_com_pr[`i',1]/sigma[`i',1]);
		matrix prob_com[`i',1] = p_c;
		scalar e_r = z_ret_es[`i',1]+sigma[`i',1]*normden(-z_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_ret_es[`i',1]/sigma[`i',1]));
		matrix es_ret[`i',1] = e_r;
		scalar e_c = z_com_es[`i',1]+sigma[`i',1]*normden(-z_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_com_es[`i',1]/sigma[`i',1]));
		matrix es_com[`i',1] = e_c;
		if `proxtemp'==0 {;
			scalar p_r_fd = normprob(z_fd_ret_pr[`i',1]/sigma[`i',1]);
			matrix prob_ret_fd[`i',1] = p_r_fd;
			scalar p_c_fd = normprob(z_fd_com_pr[`i',1]/sigma[`i',1]);
			matrix prob_com_fd[`i',1] = p_c_fd;
			scalar e_r_fd = z_fd_ret_es[`i',1]+sigma[`i',1]*normden(-z_fd_ret_es[`i',1]/sigma[`i',1])/(1-normprob(-z_fd_ret_es[`i',1]/sigma[`i',1]));
			matrix es_ret_fd[`i',1] = e_r_fd;
			scalar e_c_fd = z_fd_com_es[`i',1]+sigma[`i',1]*normden(-z_fd_com_es[`i',1]/sigma[`i',1])/(1-normprob(-z_fd_com_es[`i',1]/sigma[`i',1]));
			matrix es_com_fd[`i',1] = e_c_fd;
		};	
		
		local i = `i'+1;
	};

	/*
	convert simulation results matrices into variables
	*/
	
	svmat prob_ret, names(prob_ret_v);
	svmat prob_com, names(prob_com_v);
	svmat es_ret, names(es_ret_v);
	svmat es_com, names(es_com_v);
	
	svmat prob_ret_fd, names(prob_ret_fd_v);
	svmat prob_com_fd, names(prob_com_fd_v);
	svmat es_ret_fd, names(es_ret_fd_v);
	svmat es_com_fd, names(es_com_fd_v);

	/*
	calculate observed difference in probabilities of prison
	*/
	
	gen probdif = prob_com_v1-prob_ret_v1;

	if `proxtemp'==0 {;
		replace prob_ret_fd_v1=prob_ret_fd_v1-prob_ret_v1;
		replace prob_com_fd_v1=prob_com_fd_v1-prob_com_v1;
		};

	/*
	Calculate the change in the odds ratio
	*/
	
	gen odds_com = prob_com_v1/(1-prob_com_v1);
	gen odds_ret = prob_ret_v1/(1-prob_ret_v1);
	gen oddsdif = (odds_com-odds_ret)/odds_ret;

	/*
	Calculate the change in the sentence
	*/

	replace es_com_v1=exp(es_com_v1);
	replace es_ret_v1=exp(es_ret_v1);

	gen sentdif = es_com_v1-es_ret_v1;

	if `proxtemp'==0 {;
		replace es_ret_fd_v1=exp(es_ret_fd_v1);
		replace es_com_fd_v1=exp(es_com_fd_v1);
		replace es_ret_fd_v1=es_ret_fd_v1-es_ret_v1;
		replace es_com_fd_v1=es_com_fd_v1-es_com_v1;
	};
	
	/*
	Take 2.5, 50, and 97.5th percentile of quantities of interest
	*/
	
	qui centile prob_ret_v1, centile(2.5 50 97.5);
	qui replace prob_ret_l = r(c_1) in `obstemp';
	qui replace prob_ret_m = r(c_2) in `obstemp';
	qui replace prob_ret_h = r(c_3) in `obstemp';
	
	qui centile prob_com_v1, centile(2.5 50 97.5);
	qui replace prob_com_l = r(c_1) in `obstemp';
	qui replace prob_com_m = r(c_2) in `obstemp';
	qui replace prob_com_h = r(c_3) in `obstemp';
	
	qui centile probdif, centile(2.5 50 97.5);
	qui replace probdif_l = r(c_1) in `obstemp';
	qui replace probdif_m = r(c_2) in `obstemp';
	qui replace probdif_h = r(c_3) in `obstemp';

	qui centile odds_ret, centile(2.5 50 97.5);
	qui replace odds_ret_l = r(c_1) in `obstemp';
	qui replace odds_ret_m = r(c_2) in `obstemp';
	qui replace odds_ret_h = r(c_3) in `obstemp';
	
	qui centile odds_com, centile(2.5 50 97.5);
	qui replace odds_com_l = r(c_1) in `obstemp';
	qui replace odds_com_m = r(c_2) in `obstemp';
	qui replace odds_com_h = r(c_3) in `obstemp';

	qui centile oddsdif, centile(2.5 50 97.5);
	qui replace oddsdif_l = r(c_1) in `obstemp';
	qui replace oddsdif_m = r(c_2) in `obstemp';
	qui replace oddsdif_h = r(c_3) in `obstemp';

	qui centile es_ret_v1, centile(2.5 50 97.5);
	qui replace es_ret_l = r(c_1) in `obstemp';
	qui replace es_ret_m = r(c_2) in `obstemp';
	qui replace es_ret_h = r(c_3) in `obstemp';
	
	qui centile es_com_v1, centile(2.5 50 97.5);
	qui replace es_com_l = r(c_1) in `obstemp';
	qui replace es_com_m = r(c_2) in `obstemp';
	qui replace es_com_h = r(c_3) in `obstemp';

	qui centile sentdif, centile(2.5 50 97.5);
	qui replace sentdif_l = r(c_1) in `obstemp';
	qui replace sentdif_m = r(c_2) in `obstemp';
	qui replace sentdif_h = r(c_3) in `obstemp';
	
	if `proxtemp'==0 {;
		qui centile prob_ret_fd_v1, centile(2.5 50 97.5);
		qui replace prob_ret_fd_l = r(c_1) in `obstemp';
		qui replace prob_ret_fd_m = r(c_2) in `obstemp';
		qui replace prob_ret_fd_h = r(c_3) in `obstemp';

		qui centile prob_com_fd_v1, centile(2.5 50 97.5);
		qui replace prob_com_fd_l = r(c_1) in `obstemp';
		qui replace prob_com_fd_m = r(c_2) in `obstemp';
		qui replace prob_com_fd_h = r(c_3) in `obstemp';

		qui centile es_ret_fd_v1, centile(2.5 50 97.5);
		qui replace es_ret_fd_l = r(c_1) in `obstemp';
		qui replace es_ret_fd_m = r(c_2) in `obstemp';
		qui replace es_ret_fd_h = r(c_3) in `obstemp';

		qui centile es_com_fd_v1, centile(2.5 50 97.5);
		qui replace es_com_fd_l = r(c_1) in `obstemp';
		qui replace es_com_fd_m = r(c_2) in `obstemp';
		qui replace es_com_fd_h = r(c_3) in `obstemp';
	};	
	
	/*
	drop simulator variables for next loop
	*/

	drop prob_ret_v1-sentdif;
	replace proxhat=`proxtemp' in `obstemp';
	
	local proxloop = `proxloop' + 1;
	
};

drop b1-sigma;

/* summarize first differences shifts in electoral proximity */

/* Probability of incerceration */
l prob_ret_fd* prob_com_fd* in 1;

/* baselines */
l prob_ret_m prob_com_m in 1;

/* Sentence, given any sentence assigned */
l es_ret_fd_* es_com_fd_* in 1;

/* baselines */
l es_ret_m es_com_m in 1;

l proxhat prob_ret_l-prob_com_h if model==2;

l proxhat probdif* if model==2;

l proxhat sentdif* if model==2;

drop proxhat-model;

log close;
