/***************************************************************
* Conduct Kansas Sentencing Regressions, QJPS Version                                        
* Authors: Sanford Gordon, Gregory Huber                   
* Update: March 1, 2007                                   
* Uses:
*    QJPS_activekssentences.dta
*    QJPS_judge_ProxData.dta
*    QJPS_KansasDistrictCharacteristics.dta
*    QJPS_judgeid_sentdate_whetherjudgechallenged.dta
*
* Robustness Checks Only
*
***************************************************************/

#delimit ;
pause off ;

set mem 400m;
set seed 666;
set matsize 5000;
set more off;

log using "logs\QJPS_RobustnessChecks.log", replace;

*************
* LOAD DATA *
*************;

*load sentencing data;
use data\QJPS_activekssentences.dta;

************************
* Pre-Process KSC data *
************************;

*Drop murders;
drop if c_dead==1;
drop c_dead;

*Drop offgrid (non-guideline) crimes;
drop if off_non ~= .;

*In the KSC data there is a censoring problem with addcount, such that when addcount=5, counts can be greater than 6;
*We recode addcount to fix this;
replace addcount=(counts-1) if addcount==5 & counts>6;

*Create a binary dependent variable which is 1 if any prison time is assigned;
gen prison=0;
replace prison=1 if totpris>0;

****************************
* End Pre-Process KSC data *
****************************;

*merge in electoral proximity data by judgeid and day of sentencing;
*elecprox is scaled 0 to 1 as described in text;
*alternative measure discussed in footnote 21 is variable proximity_filingdate;
rename sentdate day;
sort judgeid day;
joinby judgeid day using data\QJPS_judge_ProxData.dta;
rename day sentdate;

*merge in district characteristics by district;
sort district;
joinby district using data\QJPS_KansasDistrictCharacteristics.dta;

*merge in data on whether judge was challenged;
*variable challenged=1 if a judge (in a partisan district) is challenged during that term;
sort judgeid sentdate;
joinby judgeid sentdate using data\QJPS_judgeid_sentdate_whetherjudgechallenged.dta;

************************************************
* Create Electoral Proximity Interaction Terms *
************************************************;

*Proximity in competitive and retention districts;
gen comprox = elecprox*(partisan);
gen ncomprox = elecprox*(1-partisan);

*Robustness checks, interactions between certain variables and electoral proximity;
gen crimerate_prox=crimerate*elecprox;
gen sctyesprop_prox=sctyesprop*elecprox;
gen turnout_prox=turnoutpres00*elecprox;

******************************
* Create Judge Fixed Effects *
******************************;

quietly tab judgeid, gen(jd_);

****************
* ANALYZE DATA *
****************;

/* ********************************************** */
/* Robustness Checks, in order discussed in paper */
/* ********************************************** */

*******************************************************************************************************************
* ROBUSTNESS, see FN 27: Table 5, columns (1) and (2), but exluding cases where judges lost/retired (e.g. prox=0) *
*******************************************************************************************************************;

/* ***************************************************************** */
/* Column (1), Model with district characteristics, no judge dummies */
/* ***************************************************************** */

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan comprox ncomprox if elecprox~=0, nocons cluster(judgeid) robust;

lincom partisan + comprox - ncomprox;
lincom partisan * 1 + comprox*.30 - ncomprox*.30;
lincom partisan * 1 + comprox*.33 - ncomprox*.33;
lincom partisan * 1 + comprox*.35 - ncomprox*.35;
test comprox==ncomprox;

/* ***************************************************************** */
/* Column (2), Model with judge dummies                              */
/* ***************************************************************** */

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 comprox ncomprox jd_70 jd_1-jd_69 jd_71-jd_135 if elecprox~=0, nocons robust;
test comprox==ncomprox;

*******************************************************************************************************************
* ROBUSTNESS, Table 3, columns (1), alternative ways of clustering                                                *
*******************************************************************************************************************;

*****************************************************************
*Column (1), basic systemic effect, no district characteristics *
*****************************************************************;

*Cluster judge-year;
gen judgeyear=year*1000+judgeid;
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan, nocons cluster(judgeyear) robust;
*Cluster district-year;
gen distyear=year*1000+district;
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan, nocons cluster(distyear) robust;
drop distyear;
*Cluster district;
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan, nocons cluster(district) robust;

*******************************************************************************************************************
* ROBUSTNESS, Table 5, columns (1), Interaction between Crime rate and proximity.                                 *
* (For dropping Shawnee, see below which drops each dsitrict once                                                 *
*******************************************************************************************************************;

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate crimerate_prox sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan comprox ncomprox, nocons cluster(judgeid) robust;
lincom partisan+comprox-ncomprox;
lincom partisan+.33*comprox-.33*ncomprox;

*******************************************************************************************************************
* ROBUSTNESS, Table 5, columns (1), Interaction between S. Ct. Retention votes and proximity                      *
* Also, as mentioned in footnote, interaction between turnout and proximity                                       *
*******************************************************************************************************************;

/* ********************************************************************************* */
/* Model with interaction between electoral proximity and avg. S. Ct. retention vote */
/* ********************************************************************************* */

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan comprox ncomprox sctyesprop_prox, nocons cluster(judgeid) robust;

*average Yes vote rate is .7495779;
lincom partisan + comprox + .7495779*sctyesprop_prox;
lincom partisan + comprox - ncomprox;
lincom partisan +.33* comprox - .33*ncomprox;

/* ************************************************************************ */
/* Model with interaction between electoral proximity and turnout rate      */
/* ************************************************************************ */

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan comprox ncomprox turnout_prox, nocons cluster(judgeid) robust;

*average turnout is .5384412;
lincom partisan+comprox+turnout_prox*.5384412;

lincom partisan+comprox-ncomprox;
lincom partisan * 1 + comprox*.30 - ncomprox*.30;
lincom partisan * 1 + comprox*.33 - ncomprox*.33;
lincom partisan * 1 + comprox*.35 - ncomprox*.35;
test comprox==ncomprox;

**********************************************************************************************************
* ROBUSTNESS, Table 5, column (2), Urban versus rural districts, by analyzing 5 urban districts sep.     *
**********************************************************************************************************;

/* urban districts */
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 jd_* comprox ncomprox if (district==29 | district==18 | district==10 | district==7 | district==3), robust;
test comprox==ncomprox;

/* rural districts */
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 jd_* comprox ncomprox if district~=29 & district~=18 & district~=10 & district~=7 & district~=3, robust;
test comprox==ncomprox;

*******************************************************************************************************************
* ROBUSTNESS, Table 5, columns (1) and (2), Dropping each district once.                                          *
*******************************************************************************************************************;

local i = 1;
while `i' <30 {;
local label = (string(`i'));
disp "dropping district "+`label';

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop y_1 y_2 y_3 y_4 y_5 y_6 y_7 partisan comprox ncomprox if district~=`i', robust cluster(judgeid);
test comprox==ncomprox;

if `i'==1 {;
outreg partisan comprox ncomprox using "logs\QJPS_RobustnessDroppingDistricts.out", addstat(log-likelihood, e(ll)) bfmt(fc) bdec(3) tdec(3) ctitle("Drop "`label') se nonotes noaster replace;
};
else {;
outreg partisan comprox ncomprox using "logs\QJPS_RobustnessDroppingDistricts.out", addstat(log-likelihood, e(ll)) bfmt(fc) bdec(3) tdec(3) ctitle("Drop "`label') se nonotes noaster append;
};

intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan age age2 y_1 y_2 y_3 y_4 y_5 y_6 y_7 jd_* comprox ncomprox if district~=`i', robust;
test comprox==ncomprox;

outreg comprox ncomprox using "logs\QJPS_RobustnessDroppingDistricts.out", addstat(log-likelihood, e(ll)) bfmt(fc) bdec(3) tdec(3) ctitle("JD, Drop "`label') se nonotes noaster append;

local i = `i'+1;
};

log close;
