/***************************************************************
* Conduct Kansas Sentencing Regressions, QJPS Version                                        
* Authors: Sanford Gordon, Gregory Huber                   
* Update: February 12, 2007                                   
* Uses:
*    QJPS_activekssentences.dta
*    QJPS_judge_ProxData.dta
*    QJPS_KansasDistrictCharacteristics.dta
*    QJPS_judgeid_sentdate_whetherjudgechallenged.dta
*
* This do file creates:
*  Table 3, columns (3) and (6)
*
***************************************************************/

/*****************************************************************************
** estimator for Robustness, described in Wooldridge 2006                   **
** http://www.msu.edu/~ec/faculty/wooldridge/current%20research/clus1aea.pdf**
** Revised: January 17, 2007                                                **
*****************************************************************************/

#delimit ;

set mem 400m;
set seed 666;
set matsize 5000;
set more off;

log using "logs\QJPS_Table3Columns3and6.log", replace;

*************
* LOAD DATA *
*************;

*load sentencing data;
use data\QJPS_activekssentences.dta;

************************
* Pre-Process KSC data *
************************;

*Drop murders;
drop if c_dead==1;
drop c_dead;

*Drop offgrid (non-guideline) crimes;
drop if off_non ~= .;

*In the KSC data there is a censoring problem with addcount, such that when addcount=5, counts can be greater than 6;
*We recode addcount to fix this;
replace addcount=(counts-1) if addcount==5 & counts>6;

*Create a binary dependent variable which is 1 if any prison time is assigned;
gen prison=0;
replace prison=1 if totpris>0;

****************************
* End Pre-Process KSC data *
****************************;

*merge in electoral proximity data by judgeid and day of sentencing;
*elecprox is scaled 0 to 1 as described in text;
*alternative measure discussed in footnote 21 is variable proximity_filingdate;
rename sentdate day;
sort judgeid day;
joinby judgeid day using data\QJPS_judge_ProxData.dta;
rename day sentdate;

*merge in district characteristics by district;
sort district;
joinby district using data\QJPS_KansasDistrictCharacteristics.dta;

*merge in data on whether judge was challenged;
*variable challenged=1 if a judge (in a partisan district) is challenged during that term;
sort judgeid sentdate;
joinby judgeid sentdate using data\QJPS_judgeid_sentdate_whetherjudgechallenged.dta;

************************************************
* Create Electoral Proximity Interaction Terms *
************************************************;

*Proximity in competitive and retention districts;
gen comprox = elecprox*(partisan);
gen ncomprox = elecprox*(1-partisan);

*Robustness checks, interactions between certain variables and electoral proximity;
gen crimerate_prox=crimerate*elecprox;
gen sctyesprop_prox=sctyesprop*elecprox;
gen turnout_prox=turnoutpres00*elecprox;
****************
* ANALYZE DATA *
****************;

*****************************************
* Create Table 3, columns 3 and 6       *
*****************************************;

/******************************************
** All Judges, column 3                  **
******************************************/

sort district;
quietly tab district, gen(dist_);
drop dist_17;

/*Note that we can't recover a coefficient on district 17, which only has 68 observations*/
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat 
       c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan 
       age y_1 y_2 y_3 y_4 y_5 y_6 y_7 dist_*, robust nocons; 

/* System matrices */
matrix betahat = get(_b);
matrix vce = get(VCE);

/* Dependent variable, unweighted, is the vector of fixed effects */
matrix y = betahat[1,"model:dist_1".."model:dist_31"]';

/* Obtain the variance of each fixed effect coefficient, in a vector */
matrix vars = vecdiag(vce);
matrix vars = vars[1,"model:dist_1".."model:dist_31"]'; 

drop if district == 17;

predict prishat if e(sample);
drop if prishat == .;
drop prishat;

/*This creates a 30 observation dataset with N = number of observations */
gen N = 1;
sort district;
collapse (sum) N, by(district);
sort district;

/*The weight on each district is equal to the ratio of the sample size and the coefficient variance */
svmat vars;
gen w_i = sqrt(N/vars); /*Note the square root, which is used to transform the data for WLS */

/*Here is the partisan system indicator for all districts except for 17*/
matrix partisan_no17 = (0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,0,1,1,1,0,1,1,0,1,0,0)';
svmat partisan_no17;

gen w_i_partisan = partisan*w_i; /*Transformed version of the partisan dummy*/

svmat y;
gen y_w_i = y1*w_i; /*Transformed version of the dependent variable */
reg y_w_i w_i_partisan w_i, nocons; /*Note that the transformed version of the constant one is just the weight*/
/*Just to be sure, here's the Stata command for WLS using the original (non square-root) weight)*/
reg y1 partisan_no171 [aweight=w_i^2];

sort district;
joinby district using "data\QJPS_KansasDistrictCharacteristics.dta", unmatched(master);

reg y1 partisan_no171 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop [aweight=w_i^2];

outreg using "logs\QJPS_Table3_Columns3and6.out", se bfmt(fc) bdec(3) 
tdec(3) ctitle("(3)") nonotes noaster replace;

matrix drop _all;
clear;

*************
* LOAD DATA *
*************;

*load sentencing data;
use data\QJPS_activekssentences.dta;

************************
* Pre-Process KSC data *
************************;

*Drop murders;
drop if c_dead==1;
drop c_dead;

*Drop offgrid (non-guideline) crimes;
drop if off_non ~= .;

*In the KSC data there is a censoring problem with addcount, such that when addcount=5, counts can be greater than 6;
*We recode addcount to fix this;
replace addcount=(counts-1) if addcount==5 & counts>6;

*Create a binary dependent variable which is 1 if any prison time is assigned;
gen prison=0;
replace prison=1 if totpris>0;

****************************
* End Pre-Process KSC data *
****************************;

*merge in data on whether judge was challenged;
*variable challenged=1 if a judge (in a partisan district) is challenged during that term;
sort judgeid sentdate;
joinby judgeid sentdate using data\QJPS_judgeid_sentdate_whetherjudgechallenged.dta;

****************
* ANALYZE DATA *
****************;

/*********************************************
**Unchallenged judges only, column 6        **
**********************************************/

drop if challenged==1;
quietly tab district, gen(dist_);
drop dist_17;

/*Note that we can't recover a coefficient on district 17, which only has 68 observations*/
intreg lntotprisl lntotprisr ln_gpres0_pris grid1 grid2 addcount pso firearm c_assault c_crimthreat 
       c_robbery c_sex c_theft c_burglary v_govoff v_child appcouns pleabarg nonwhite male hispan 
       age y_1 y_2 y_3 y_4 y_5 y_6 y_7 dist_*, robust nocons; 

/* System matrices */
matrix betahat = get(_b);
matrix vce = get(VCE);

/* Dependent variable, unweighted, is the vector of fixed effects */
matrix y = betahat[1,"model:dist_1".."model:dist_31"]';

/* Obtain the variance of each fixed effect coefficient, in a vector */
matrix vars = vecdiag(vce);
matrix vars = vars[1,"model:dist_1".."model:dist_31"]'; 

drop if district == 17;

predict prishat if e(sample);
drop if prishat == .;
drop prishat;

/*This creates a 30 observation dataset with N = number of observations */
gen N = 1;
sort district;
collapse (sum) N, by(district);
sort district;

/*The weight on each district is equal to the ratio of the sample size and the coefficient variance */
svmat vars;
gen w_i = sqrt(N/vars); /*Note the square root, which is used to transform the data for WLS */

/*Here is the partisan system indicator for all districts except for 17*/
matrix partisan_no17 = (0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,0,1,1,1,0,1,1,0,1,0,0)';
svmat partisan_no17;

gen w_i_partisan = partisan*w_i; /*Transformed version of the partisan dummy*/

svmat y;
gen y_w_i = y1*w_i; /*Transformed version of the dependent variable */
#delimit;
reg y_w_i w_i_partisan w_i, nocons; /*Note that the transformed version of the constant one is just the weight*/
/*Just to be sure, here's the Stata command for WLS using the original (non square-root) weight)*/
reg y1 partisan_no171 [aweight=w_i^2];

sort district;
joinby district using "data\QJPS_KansasDistrictCharacteristics.dta", unmatched(master);

reg y1 partisan_no171 turnoutpres00 presd00prop nwrate urbanrate crimerate sctyesprop [aweight=w_i^2];

outreg using "logs\QJPS_Table3_Columns3and6.out", se bfmt(fc) bdec(3) 
tdec(3) ctitle("(6)") nonotes noaster append;

matrix drop _all;

log close;
